/*
 *  V4L2 video capture example
 *
 *  This program can be used and distributed without restrictions.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>             /* getopt_long() */
#include <fcntl.h>              /* low-level i/o */
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <asm/types.h>          /* for videodev2.h */
#include <libv4l2.h>
#include <linux/videodev2.h>

#include <SDL/SDL.h>
#include <iostream>

using namespace std;

#define CLEAR(x) memset (&(x), 0, sizeof (x))

/////////////////////////////////////////////////////////
// The following are SDL yuv overlay format vs. v4l2 format

///////////////////////////////////////////////////////////////////////////
//#define SDL_YV12_OVERLAY  0x32315659  /* Planar mode: Y + V + U */
//#define V4L2_PIX_FMT_YVU420  v4l2_fourcc('Y', 'V', '1', '2') /* 12  YVU 4:2:0     */
///////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////
//#define SDL_IYUV_OVERLAY  0x56555949  /* Planar mode: Y + U + V */
// ???? It should be I420 or IYUV, but we cannot find in v4l2
////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
//#define SDL_YUY2_OVERLAY  0x32595559  /* Packed mode: Y0+U0+Y1+V0 */
//#define V4L2_PIX_FMT_YUYV    v4l2_fourcc('Y', 'U', 'Y', 'V') /* 16  YUV 4:2:2     */
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
//#define SDL_UYVY_OVERLAY  0x59565955  /* Packed mode: U0+Y0+V0+Y1 */
//#define V4L2_PIX_FMT_UYVY    v4l2_fourcc('U', 'Y', 'V', 'Y') /* 16  YUV 4:2:2     */
/////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
//#define SDL_YVYU_OVERLAY  0x55595659  /* Packed mode: Y0+V0+Y1+U0 */
//#define V4L2_PIX_FMT_YVYU    v4l2_fourcc('Y', 'V', 'Y', 'U') /* 16 YVU 4:2:2 */
//////////////////////////////////////////////////////////////////////


static char  dev_name[32];
static int fd                        = -1;
static struct v4l2_format fmt;
static SDL_Surface* screen           = NULL;
static SDL_Overlay* displayOverlay   = NULL;
static SDL_Rect rect;


struct buffer
{
    void   *start;
    size_t length;
};

#define SDL_FORMAT_NUMBER 3
unsigned long SDL_Formats[SDL_FORMAT_NUMBER] = {SDL_YUY2_OVERLAY, SDL_UYVY_OVERLAY, SDL_YVYU_OVERLAY};
unsigned long V4L2_Formats[SDL_FORMAT_NUMBER] = {V4L2_PIX_FMT_YUYV, V4L2_PIX_FMT_UYVY, V4L2_PIX_FMT_YVYU};
unsigned int formatIndex = 0;
#define REQUEST_BUFFER_COUNT 2
static struct buffer buffers[REQUEST_BUFFER_COUNT];


static int xioctl(int fd, int request, void* arg)
{
    int r;
    do
    {
    	r = ioctl (fd, request, arg);
    }
    while (-1 == r && EINTR == errno);

    return r;
}

static void mainloop()
{
#define FPS_COUNT_NUMBER 30
	SDL_Event event;
	struct v4l2_buffer buf;
	struct timeval last, now;
	int counter = 0;
    while (true)
	{
		if (counter == 0)
		{
			gettimeofday(&last, NULL);
		}
		SDL_PollEvent(&event);
		if (event.type == SDL_QUIT)
		{
			break;
		}
		int r;
		do
        {
        	struct timeval tv;
			fd_set fds;
            FD_ZERO(&fds);
            FD_SET(fd, &fds);

            /* Timeout. */
            tv.tv_sec = 1;
            tv.tv_usec = 0;

            r = select(fd + 1, &fds, NULL, NULL, &tv);
        }
        while ((r == -1 && (errno = EINTR)));
        if (r == -1)
        {
            perror("select");
            break;
        }
        CLEAR(buf);
        buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        buf.memory = V4L2_MEMORY_MMAP;
        if (xioctl(fd, VIDIOC_DQBUF, &buf) == -1)
        {
        	break;
        }

		if (SDL_LockYUVOverlay(displayOverlay) == -1)
		{
			fprintf (stderr, "SDL lock overlay error\n");
			break;
		}
		memcpy(displayOverlay->pixels[0], buffers[buf.index].start, buf.bytesused);

		SDL_UnlockYUVOverlay(displayOverlay);
		if (xioctl(fd, VIDIOC_QBUF, &buf) == -1)
		{
			break;
		}
		if (SDL_DisplayYUVOverlay(displayOverlay, &rect) == -1)
		{
			fprintf (stderr, "SDL display overlay error\n");
			break;
		}

		counter ++;
		if (counter == FPS_COUNT_NUMBER)
		{
			gettimeofday(&now, NULL);
			double diff = (now.tv_sec - last.tv_sec) + (double)(now.tv_usec - last.tv_usec)/1000000.0;

			printf("fps:%f\n", (double)FPS_COUNT_NUMBER/diff);
			counter = 0;
		}
    }
}

static bool init_buffer()
{
	struct v4l2_buffer              buf;
    struct v4l2_requestbuffers      req;
    bool bHasError = false;
    int n_buffers;
    enum v4l2_buf_type              type;

	CLEAR(req);
	CLEAR(buffers);
    req.count = REQUEST_BUFFER_COUNT;
    req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    req.memory = V4L2_MEMORY_MMAP;
    if (xioctl(fd, VIDIOC_REQBUFS, &req) == -1)
    {
    	return false;
    }

    for (n_buffers = 0; n_buffers < req.count; ++ n_buffers)
    {
        CLEAR(buf);

        buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        buf.memory      = V4L2_MEMORY_MMAP;
        buf.index       = n_buffers;

        if (xioctl(fd, VIDIOC_QUERYBUF, &buf) == -1)
        {
        	bHasError = true;
        	break;
        }

        buffers[n_buffers].length = buf.length;
        buffers[n_buffers].start = v4l2_mmap(NULL, buf.length,
                                             PROT_READ | PROT_WRITE, MAP_SHARED,
                                             fd, buf.m.offset);

        if (MAP_FAILED == buffers[n_buffers].start)
        {
            bHasError = true;
            break;
        }
        if (xioctl(fd, VIDIOC_QBUF, &buf) == -1)
        {
        	bHasError = true;
            break;
        }
    }
    if (bHasError)
    {
    	for (int i = 0; i < n_buffers; i ++)
    	{
    		if (buffers[i].start)
    		{
    			v4l2_munmap(buffers[i].start, buffers[i].length);
    		}
    	}
    	return false;
    }

    type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

    if (xioctl(fd, VIDIOC_STREAMON, &type) == -1)
    {
    	for (int i = 0; i < n_buffers; i ++)
    	{
    		if (buffers[i].start)
    		{
    			v4l2_munmap(buffers[i].start, buffers[i].length);
    		}
    	}
    	return false;
    }
    return true;
}


static bool init_read(const v4l2_pix_format& fmt)
{
	bool bResult = false;
    if ((screen = SDL_SetVideoMode(fmt.width, fmt.height, 24, SDL_SWSURFACE))!= NULL)
    {
		if ((displayOverlay = SDL_CreateYUVOverlay(fmt.width, fmt.height, SDL_Formats[formatIndex], screen)) != NULL)
		{
			if (displayOverlay->pitches[0] == fmt.bytesperline)
			{
				if (init_buffer())
				{
					rect.x = rect.y = 0;
					rect.w = fmt.width;
					rect.h = fmt.height;
					SDL_WM_SetCaption("V4L2+SDL video capture", NULL);
					bResult = true;
				}
			}
			if (!bResult)
			{
				SDL_FreeYUVOverlay(displayOverlay);
			}
		}
		if (!bResult)
		{
			SDL_FreeSurface(screen);
		}
    }
    return bResult;
}

static bool init_device()
{
    struct v4l2_capability cap;

    if (-1 == xioctl (fd, VIDIOC_QUERYCAP, &cap))
    {
        return false;
    }

    if (!(cap.capabilities & V4L2_CAP_VIDEO_CAPTURE))
    {
        return false;
    }

    if (!(cap.capabilities & V4L2_CAP_STREAMING))
    {
        return false;
    }

	// try to enum to find matching format
	// currently there are five SDL supported format

	//unsigned long SDL_Formats[SDL_FORMAT_NUMBER] = {SDL_YUY2_OVERLAY, SDL_UYVY_OVERLAY, SDL_YVYU_OVERLAY};

	int index = 0;
	bool bFoundMatch = false;
	do
	{
		v4l2_fmtdesc desc;
		desc.index = index;
		desc.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

		if (xioctl(fd, VIDIOC_ENUM_FMT, &desc) == -1)
		{
			break;
		}
		for (int i = 0; i < SDL_FORMAT_NUMBER; i ++)
		{
			if (desc.pixelformat == V4L2_Formats[i])
			{
				bFoundMatch = true;
				CLEAR (fmt);
				fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
				fmt.fmt.pix.pixelformat = desc.pixelformat;
				formatIndex = i;
				fmt.fmt.pix.width = 640;
				fmt.fmt.pix.height = 480;
				break;
			}
		}
		index ++;
	}
	while (!bFoundMatch);

	if (bFoundMatch)
	{
		if (-1 == xioctl (fd, VIDIOC_S_FMT, &fmt))
		{
			printf("error %s\n", strerror(errno));
			return false;
		}
	}

    return init_read(fmt.fmt.pix);
}

static void close_device()
{
	enum v4l2_buf_type type;

	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    xioctl(fd, VIDIOC_STREAMOFF, &type);
    for (int i = 0; i < REQUEST_BUFFER_COUNT; ++ i)
    {
        v4l2_munmap(buffers[i].start, buffers[i].length);
    }

	SDL_FreeSurface(screen);
	SDL_FreeYUVOverlay(displayOverlay);
	v4l2_close(fd);
}

static bool open_device()
{
	for (int i = 0; i < 64; i ++)
	{
		struct stat st;
		sprintf(dev_name, "/dev/video%d", i);
		if (0 == stat (dev_name, &st))
		{
			if (S_ISCHR (st.st_mode))
			{
				if ((fd = v4l2_open(dev_name, O_RDWR | O_NONBLOCK, 0)) != -1)
				{
					if (init_device())
					{
						return true;
					}
					close(fd);
				}
			}
		}
	}
	return false;
}

int main(int argc, char** argv)
{
	if ( SDL_Init(SDL_INIT_EVERYTHING) == -1 )
    {
        fprintf (stderr, "SDL Init error\n");
        return -1;
    }

    if (open_device())
    {
		mainloop();
		close_device();
    }

	SDL_Quit();
    return 0;
}

