// CSS
//
// complete CSS implementation
//
// see http://www.tinyted.net/eddie/css.html for details
//
// this file is hereby placed into the public domain, and can
// be used by anyone for any purpose
//
// the complete implementation is three files:
//
// css.cpp
// css.h [this file]
// css_tables.h
//
// I would appreciate an email [eddie@tinyted.net] if you
// are using this code

extern unsigned char reverse[256];

class LFSR1
{
public:
	inline LFSR1(unsigned long value);			// initialize directly
	inline LFSR1(const unsigned char key[5]);	// initialize from key

	inline bool	Write(unsigned char key[5]);	// write back to key - returns false if
												// LFSR1 is not in a good initialized state

	inline void	Step1();
	inline void	Step8();
	inline void	Back1();
	inline void	Back8();

	unsigned long	GetValue()		{ return val; }

	unsigned char	GetByte()		{ return unsigned char(val >> 9); }

private:
	unsigned long	val;	
};

class LFSR2
{
public:
	inline LFSR2(unsigned long value);			// initialize directly
	inline LFSR2(const unsigned char key[5]);	// initialize from key

	inline bool	Write(unsigned char key[5]);	// write back to key - returns false if
												// LFSR2 is not in a good initialized state

	inline void	Step1();
	inline void	Step8();
	inline void	Back1();
	inline void	Back8();

	unsigned long	GetValue()		{ return val; }

	unsigned char	GetByte()		{ return unsigned char(val >> 17); }

	inline void	FeedByte(unsigned char byte);

private:
	unsigned long	val;	
};

// implementation (inline for speed)

// direct initialization

LFSR1::LFSR1(unsigned long value)
{
	val = value & 0x1FFFF;
}

LFSR2::LFSR2(unsigned long value)
{
	val = value & 0x1FFFFFF;
}

// init from key

LFSR1::LFSR1(const unsigned char key[5])
{
	val = (reverse[key[0]] << 9)
		| 0x100
		| reverse[key[1]];
}

LFSR2::LFSR2(const unsigned char key[5])
{
	val = (reverse[key[2] & 0x07] << 17)
		| 0x200000
		| (reverse[key[2] & 0xF8] << 16)
		| (reverse[key[3]] << 8)
		| reverse[key[4]];
}

// write to key

bool LFSR1::Write(unsigned char key[5])
{
	if (!(val & 0x100))	return false;

	key[0] = reverse[val >> 9];
	key[1] = reverse[val & 0xFF];
}

bool LFSR2::Write(unsigned char key[5])
{
	if (!(val & 0x200000))	return false;

	key[2] = reverse[((val >> 16) & 0x1F) | ((val >> 17) & 0xE0)];
	key[3] = reverse[(val >> 8) & 0xFF];
	key[4] = reverse[val & 0xFF];
}

// step forward 1 bit

void LFSR1::Step1()
{
	unsigned long	bit = (val ^ (val >> 14)) & 1;

	val = (val >> 1) | (bit << 16);
}

void LFSR2::Step1()
{
	unsigned long	bit = (val ^ (val >> 3) ^ (val >> 4) ^ (val >> 12)) & 1;

	val = (val >> 1) | (bit << 24);
}

// step forward 8 bits

void LFSR1::Step8()
{
	val = (val << 9) | (val >> 8);
	unsigned long bits = val & 0x03FC0;
	unsigned long product = (bits << 3) ^ (bits << 6) ^ (bits << 9);
	val ^= product;
	val &= 0x1FFFF;
}

void LFSR2::Step8()
{
	unsigned long left8 = val ^ (val >> 3) ^ (val >> 4) ^ (val >> 12);
	val = (left8 << 17) | (val >> 8);
	val &= 0x1FFFFFF;
}

// step back 1 bit

void LFSR1::Back1()
{
	unsigned long bit = ((val >> 16) ^ (val >> 13)) & 1;

	val = ((val << 1) | bit) & 0x1FFFF;
}

void LFSR2::Back1()
{
	unsigned long	bit = ((val >> 24) ^ (val >> 2) ^ (val >> 3) ^ (val >> 11)) & 1;

	val = ((val << 1) | bit) & 0x1FFFFFF;
}

// step back 8 bits

void LFSR1::Back8()
{
	unsigned long term2 = (val >> 6) & 0xFF;
	val = (val << 8) | (val >> 9);
	val ^= term2;
	val &= 0x1FFFF;
}

void LFSR2::Back8()
{
	val = (val << 8) | (val >> 17);
	unsigned long product = ((val >> 3) ^ (val >> 4) ^ (val >> 12)) & 0xFF;
	val ^= product ^ (product >> 3) ^ (product >> 4) ^ (product >> 6);
	val &= 0x1FFFFFF;
}

// accept byte from plaintext attack

void LFSR2::FeedByte(unsigned char byte)
{
	val = (val >> 8) | (byte << 17);
}

class LFSRBYTE
{
public:
	inline LFSRBYTE(unsigned long _lfsr1, unsigned long _lfsr2, unsigned long _cc);
	inline LFSRBYTE(const unsigned char key[5]);

	inline unsigned char	GetNext(unsigned char H1, unsigned char H2);
	inline void				FeedNext(unsigned char byte, unsigned char H1, unsigned char H2);
	inline void				StepBack();

private:
	LFSR1			lfsr1;
	LFSR2			lfsr2;
	unsigned long	cc;
};

LFSRBYTE::LFSRBYTE(unsigned long _lfsr1, unsigned long _lfsr2, unsigned long _cc) : lfsr1(_lfsr1), lfsr2(_lfsr2)
{
	cc = _cc & 1;
}

LFSRBYTE::LFSRBYTE(const unsigned char key[5]) : lfsr1(key), lfsr2(key)
{
	cc = 0;
}

unsigned char LFSRBYTE::GetNext(unsigned char H1, unsigned char H2)
{
	lfsr1.Step8();
	lfsr2.Step8();
	cc = (lfsr1.GetByte() ^ H1) + (lfsr1.GetByte() ^ H2) + cc;

	unsigned char	rv = cc & 0xFF;

	cc >>= 8;

	return rv;
}

void LFSRBYTE::FeedNext(unsigned char byte, unsigned char H1, unsigned char H2)
{
	lfsr1.Step8();
	cc = (byte - (lfsr1.GetByte() ^ H1) - cc) ^ H2;
	lfsr2.FeedByte(cc & 0xFF);
	cc = (cc >> 8) & 1;
}

void LFSRBYTE::StepBack()
{
	lfsr1.Back8();
	lfsr2.Back8();
	cc = 1;
}

enum CSS_Initial
{
	InitLFSR1 = 0x100,
	InitLFSR2 = 0x200000
};

class CSS_State
{
public:
	CSS_State(int _LFSR1 = InitLFSR1, int _LFSR2 = InitLFSR2, int _carry = 0)	{ LFSR1 = _LFSR1; LFSR2 = _LFSR2; carry = _carry; }

	void	Fwd1();
	void	Fwd2();
	void	Back();
	void	SetCarry(int _carry);

	void	Salt(const unsigned char salt[5]);

	void	GetKey(unsigned char key[5]) const;
	void	SetKey(const unsigned char key[5]);

	void	StepAttack(unsigned char cipher, unsigned char plain);

	unsigned char DecryptByte(unsigned char cipher, unsigned char invert0, unsigned char invert1);
	unsigned char PeekChar(unsigned char cipher) const;

	bool	IsValidInitKey() const;

private:
	int		LFSR1;
	int		LFSR2;
	int		carry;
};

class CSS_Key
{
public:
	CSS_Key()	{ }

	void	PrintKey(FILE* fd) const;

	bool	Attack(const unsigned char ciphertext[7], int offset, const unsigned char plaintext[7], const unsigned char salt[5]);

	void	Decrypt(unsigned char *buffer, const unsigned char salt[5]) const;

	void	GetKey(unsigned char _key[5]) const		{ key.GetKey(_key); }
	void	SetKey(const unsigned char _key[5])		{ key.SetKey(_key); }

private:
	CSS_State	key;
};

enum CSS_KeyType
{
	CSS_KEY1 = 0,
	CSS_KEY2 = 1,
	CSS_BUSKEY = 2
};

extern void CSS_CryptKey(int variant, int key_type, const unsigned char challenge[10], unsigned char response[5]);
