// CSS
//
// complete CSS implementation
//
// see http://www.tinyted.net/eddie/css.html for details
//
// this file is hereby placed into the public domain, and can
// be used by anyone for any purpose
//
// the complete implementation is three files:
//
// css.cpp [this file]
// css.h
// css_tables.h
//
// I would appreciate an email [eddie@tinyted.net] if you
// are using this code

#include <stdio.h>

#include "css.h"
#include "css_tables.h"

// Fwd1
//
// step LFSR1 forwards

void CSS_State::Fwd1()
{
	LFSR1 = (LFSR1 << 9) | (LFSR1 >> 8);
	unsigned long bits = LFSR1 & 0x03FC0;
	unsigned long product = (bits << 3) ^ (bits << 6) ^ (bits << 9);
	LFSR1 ^= product;
	LFSR1 &= 0x1FFFF;
}

// Fwd2
//
// step LFSR2 forwards

void CSS_State::Fwd2()
{
	unsigned long left8 = LFSR2 ^ (LFSR2 >> 3) ^ (LFSR2 >> 4) ^ (LFSR2 >> 12);
	LFSR2 = (left8 << 17) | (LFSR2 >> 8);
	LFSR2 &= 0x1FFFFFF;
}

// Back
//


void CSS_State::Back()
{
	unsigned long term2 = (LFSR1 >> 6) & 0xFF;
	LFSR1 = (LFSR1 << 8) | (LFSR1 >> 9);
	LFSR1 ^= term2;
	LFSR1 &= 0x1FFFF;

	LFSR2 = (LFSR2 << 8) | (LFSR2 >> 17);
	unsigned long product = ((LFSR2 >> 3) ^ (LFSR2 >> 4) ^ (LFSR2 >> 12)) & 0xFF;
	LFSR2 ^= product ^ (product >> 3) ^ (product >> 4) ^ (product >> 6);
	LFSR2 &= 0x1FFFFFF;
}

void CSS_State::SetCarry(int _carry)
{
	carry = _carry;
}

void CSS_State::Salt(const unsigned char salt[5])
{
	LFSR1 ^= (reverse[salt[0]] << 9) | reverse[salt[1]];
	LFSR2 ^= ((reverse[salt[2]] & 0xE0) << 17) | ((reverse[salt[2]] & 0x1F) << 16) | (reverse[salt[3]] << 8) | reverse[salt[4]];
	carry = 0;
}

void CSS_State::GetKey(unsigned char key[5]) const
{
	key[0] = reverse[LFSR1 >> 9];
	key[1] = reverse[LFSR1 & 0xFF];
	key[2] = reverse[((LFSR2 >> 16) & 0x1F) | ((LFSR2 >> 17) & 0xE0)];
	key[3] = reverse[(LFSR2 >> 8) & 0xFF];
	key[4] = reverse[LFSR2 & 0xFF];
}

void CSS_State::SetKey(const unsigned char key[5])
{
	LFSR1 = InitLFSR1;
	LFSR2 = InitLFSR2;
	Salt(key);
}

void CSS_State::StepAttack(unsigned char cipher, unsigned char plain)
{
	Fwd1();
	carry = (css_table[cipher] ^ plain) - ((LFSR1 >> 9) ^ 0xFF) - carry;
	LFSR2 = (LFSR2 >> 8) | ((carry & 0xFF) << 17);
	carry = (carry >> 8) & 0x01;
}

unsigned char CSS_State::DecryptByte(unsigned char cipher, unsigned char invert0, unsigned char invert1)
{
	Fwd1();
	Fwd2();
	carry += ((LFSR1 >> 9) ^ invert0) + ((LFSR2 >> 17) ^ invert1);

	unsigned char plain = css_table[cipher] ^ carry;

	carry >>= 8;

	return plain;
}

unsigned char CSS_State::PeekChar(unsigned char cipher) const
{
	int	ncarry = ((LFSR1 >> 9) ^ 0xFF) + (LFSR2 >> 17) + carry;
	return css_table[cipher] ^ ncarry;
}

bool CSS_State::IsValidInitKey() const
{
	return (LFSR1 & 0x100) && (LFSR2 & 0x200000);
}

void CSS_Key::PrintKey(FILE* fd) const
{
	unsigned char	skey[5];

	key.GetKey(skey);

	fprintf(fd, "%2.2X %2.2X %2.2X %2.2X %2.2X", skey[0], skey[1], skey[2], skey[3], skey[4]);
}

// Attack
//
// attack the CSS cipher using 7 known bytes
//
// ciphertext is the encrypted part value
// offset is the offset into the encrypted data of the ciphertext
// plaintext is the corresponding decrypted value
// salt is the salt for the encryption

bool CSS_Key::Attack(const unsigned char ciphertext[7], int offset, const unsigned char plaintext[7], const unsigned char salt[5])
{
	CSS_State	good_key;
	int			count = 0;

	for (int ii = 0; ii < 0x40000; ii++)
	{
		CSS_State	css(ii >> 1, 0, ii & 1);

		for (int jj = 0; jj < 4; jj++)
		{
			css.StepAttack(ciphertext[jj], plaintext[jj]);
		}
		for (; jj < 7; jj++)
		{
			if (css.DecryptByte(ciphertext[jj], 0xFF, 0x00) != plaintext[jj])	break;
		}
		if (jj < 7)	continue;

		for (jj = 0; jj < 6; jj++)
		{
			css.Back();
		}
		css.SetCarry(ii & 1);
		if (css.PeekChar(ciphertext[0]) == plaintext[0])
		{
			for (jj = 0; jj <= offset; jj++)
			{
				css.Back();
			}
			if (css.IsValidInitKey())
			{
				css.Salt(salt);
				good_key = css;
				count++;
			}
		}
	}

	if (count == 1)
	{
		key = good_key;
		return true;
	}

	return false;
}

// Decrypt
//
// decrypt a CSS block

void CSS_Key::Decrypt(unsigned char *buffer, const unsigned char salt[5]) const
{
	CSS_State	css = key;

	css.Salt(salt);

	for (int ii = 0; ii < 0x780; ii++)
	{
		buffer[ii] = css.DecryptByte(buffer[ii], 0xFF, 0x00);
	}
}

// MixBytes
//
// do a "mix bytes" operation

static void MixBytes(CSS_State& css, unsigned char cse, unsigned char data[5])
{
	unsigned char	last = 0;

	for (int ii = 0; ii < 5; ii++)
	{
		unsigned char	a = data[ii];
		unsigned char	b = css.DecryptByte(0x33, 0xFF, 0xFF);

		b ^= a;
		b = css_mix[0][b];
		b ^= cse;
		b = css_mix[1][b];
		b ^= last;

		data[ii] = b;
		last = a;
	}
}

// MessBytes
//
// do a "mess bytes" operation

static void MessBytes(unsigned char data[5])
{
	for (int ii = 0; ii < 5; ii++)
	{
		data[ii] = css_mess[data[ii]];
	}
}

// CSS_CryptKey
//
// calculate a key response from a challenge

void CSS_CryptKey(int variant, int key_type, const unsigned char challenge[10], unsigned char response[5])
{
	unsigned char	cse;
	unsigned char	key[5];
	unsigned char*	pc = css_permute_challenge[key_type];

	// create CSS key and initialize response
	for (int ii = 0; ii < 5; ii++)
	{
		key[ii] = challenge[pc[ii]] ^ css_secret[ii];
		response[ii] = challenge[pc[ii + 5]];
	}

	CSS_State	css;
	
	css.SetKey(key);

	cse = css_variant[css_translate_variant[key_type][variant]];

	// mix and mess to get the right result
	MixBytes(css, cse, response);
	response[0] ^= response[4];
	MixBytes(css, cse, response);
	response[0] ^= response[4];
	MixBytes(css, cse, response);
	MessBytes(response);
	response[0] ^= response[4];
	MixBytes(css, cse, response);
	MessBytes(response);
	response[0] ^= response[4];
	MixBytes(css, cse, response);
	response[0] ^= response[4];
	MixBytes(css, cse, response);
}
