// SCSI_CDVD
//
// CD or DVD player

#include <stdafx.h>

#include "aspi.h"
#include "cdvd.h"

#include "css.h"

// new SCSI_CDVD
//
// determine if it's a DVD, and if so, unlocks it

SCSI_CDVD::SCSI_CDVD(SCSIDevice* dev)
{
	mDev = dev;

	// perform MODE SENSE
	SCSICmd10	cmd(SCSI_MODE_SEN10, 34);

	cmd[2] = 0x2A;	// CDVD capabilities

	u8	res[34];

	if (dev->SendCommandIn(cmd, res, 34))
	{
		u8*	info = &res[8];

		if (info[2] & 0x10)
		{
			// unit supports DVD-ROM read

			// match region
			if (MatchDriveRegionToDisk(false))
			{
				// authenticate the drive
				AuthenticateDisk();
			}
		}
	}
}

// GetRegionCode
//
// get the device's region code

u8 SCSI_CDVD::GetRegionCode()
{
	SCSICmd12	cmd(REPORT_KEY, 8);

	cmd[10] = GET_RPC_STATE;

	u8	res[8];

	if (mDev->SendCommandIn(cmd, res, 8))
	{
		int		user_resets = res[4] & 7;
		int		vendor_resets = (res[4] >> 3) & 7;

		return	res[5];
	}

	return 0xFF;
}

// SetRegionCode
//
// set the region code
// returns the updated setting

u8 SCSI_CDVD::SetRegionCode(u8 mask)
{
	// first, try and reset RPC2
	// this must be done first, so the DVD drive is properly region-set
	ResetRPC2();

	SCSICmd12	cmd(SEND_KEY, 8);

	cmd[10] = SET_RPC_STATE;

	u8	data[8];

	data[0] = 0;
	data[1] = 6;
	data[2] = 0;
	data[3] = 0;
	data[4] = mask;
	data[5] = 0;
	data[6] = 0;
	data[7] = 0;

	mDev->SendCommandOut(cmd, data, 8);

	return GetRegionCode();
}

// ResetRPC2
//
// reset RPC2 on drive

bool SCSI_CDVD::ResetRPC2()
{
	const char*	drive = "HITACHI DVD-ROM GD-7000 ";

	if (!memcmp(mDev->GetVendorID(), drive, strlen(drive)))
	{
		// it's the GD-7000 so perform the hack
		SCSICmd12	cmd(0xE7, 0);

		cmd[1] = 'H';
		cmd[2] = 'I';
		cmd[3] = 'T';
		cmd[4] = 'R';
		cmd[5] = 'P';
		cmd[6] = 'C';
		cmd[7] = '2';
		cmd[8] = 0x02;
		cmd[9] = 0x02;	// RPC2
		cmd[10] = 0x01;
		cmd[11] = 0x00;

		return mDev->SendCommandNoData(cmd);
	}

	return false;
}

// GetDiskRegionCode
//
// get the region code of the DVD

u8 SCSI_CDVD::GetDiskRegionCode()
{
	// do a READ DVD STRUCTURE
	SCSICmd12	cmd(READ_DVD_STRUCTURE, 8);

	cmd[7] = GET_DISK_COPYRIGHT;

	u8	data[8];

	if (mDev->SendCommandIn(cmd, data, 8))
	{
		return data[5];
	}

	return 0xFF;
}

// MatchDriveRegionToDisk
//
// compare the drive and disk regions
// change the drive's region to match the disk if requested

bool SCSI_CDVD::MatchDriveRegionToDisk(bool set_region)
{
	u8	rc = GetRegionCode();
	u8	drc = GetDiskRegionCode();

	if ((rc | drc) == 0xFF)
	{
		// region codes do not match
		if (!set_region)	return false;

		// find a good code
		int	region = -1;
		for (int ii = 0; ii < 8; ii++)
		{
			if (!(drc & (1 << ii)))
			{
				region = ii;
				break;
			}
		}

		if (region == -1)
		{
			// can't set region code
			return false;
		}
		else
		{
			// set matching region
			rc = SetRegionCode(0xFF - (1 << ii));
			if ((rc | drc) == 0xFF)
			{
				// failed
				return false;
			}
		}
	}

	return true;
}

// ResetCSSAuth
//
// uninitialize any existing CSS procedures on the drive

void SCSI_CDVD::ResetCSSAuth()
{
	// invalidate all four AGIDs
	for (int ii = 0; ii < 256; ii += 64)
	{
		SCSICmd12	cmd(REPORT_KEY, 0);

		cmd[10] = INVALIDATE_AGID | ii;

		mDev->SendCommandNoData(cmd);
	}
}

// IsAuthenticated
//
// check if the DVD drive is authenticated

bool SCSI_CDVD::IsAuthenticated()
{
	// check if drive is authenticated
	// do a REPORT KEY
	SCSICmd12	cmd(REPORT_KEY, 8);

	cmd[10] = GET_ASF;

	u8	data[8];

	if (mDev->SendCommandIn(cmd, data, 8))
	{
		return bool(data[7] & 1);
	}

	return false;
}

// AuthenticateDisk
//
// authenticate the DVD drive
//
// DVD authentication procedure [from Mt Fuji doc]:
//
// 1. REPORT KEY	Request AGID
// 2. SEND KEY		Send challenge key
// 3. REPORT KEY	Key 1
// 4. REPORT KEY	Challenge key
// 5. SEND KEY		Key 2
// 6. REPORT KEY	Disk or title key

bool SCSI_CDVD::AuthenticateDisk()
{
	// already authenticated?
	if (IsAuthenticated())	return true;

	// reset AGIDs on drive and request a new one
	ResetCSSAuth();

	int	agid = RequestAGID();

	if (agid == -1)			return false;

	// send challenge and determine CSS variant
	u8	challenge[10];

	int	variant = ChallengeDriveAndGetVariant(agid, challenge + 5);

	if (variant == -1)		return false;

	// receive challenge and respond
	if (!AcceptChallenge(agid, variant, challenge))	return false;

	// derive the bus key
	u8	bus_key[5];

	CSS_CryptKey(variant, CSS_BUSKEY, challenge, bus_key);

	// get the disk key
	if (!GetDiskKey(agid, bus_key))	return false;

	// final check
	return IsAuthenticated();
}

// RequestAGID
//
// get a CSS authentication ID

int	SCSI_CDVD::RequestAGID()
{
	// do a REPORT KEY
	SCSICmd12	cmd(REPORT_KEY, 8);

	cmd[10] = GET_AGID;

	u8	data[8];

	if (mDev->SendCommandIn(cmd, data, 8))
	{
		return data[7] & 0xC0;
	}

	return -1;
}

// ChallengeDriveAndGetVariant
//
// challenge the drive with a simple code
// from its response, determine which CSS variant it is using

int SCSI_CDVD::ChallengeDriveAndGetVariant(int agid, u8 key1[5])
{
	u8	challenge[10];

	for (int ii = 0; ii < 10; ii++)		challenge[ii] = ii;

	SCSICmd12	cmd(SEND_KEY, 16);

	cmd[10] = SEND_CHALLENGE_KEY | agid;

	u8	data[16];

	data[0] = 0;
	data[1] = 14;
	data[2] = 0;
	data[3] = 0;
	for (/*int*/ ii = 0; ii < 10; ii++)		data[ii + 4] = challenge[ii];
	data[14] = 0;
	data[15] = 0;

	if (!mDev->SendCommandOut(cmd, data, 16))	return -1;

	cmd = SCSICmd12(REPORT_KEY, 12);

	cmd[10] = GET_KEY1 | agid;

	if (!mDev->SendCommandIn(cmd, data, 12))	return -1;

	memcpy(key1, data + 4, 5);

	for (/*int*/ ii = 0; ii < 32; ii++)
	{
		u8	keycheck[5];
		
		CSS_CryptKey(ii, CSS_KEY1, challenge, keycheck);

		if (!memcmp(keycheck, key1, 5))
		{
			return ii;
		}
	}

	return -1;
}

// AcceptChallenge
//
// get the encrypted challenge from the drive and give the
// correct response

bool SCSI_CDVD::AcceptChallenge(int agid, int variant, u8 key[5])
{
	// get challenge from drive
	SCSICmd12	cmd(REPORT_KEY, 16);

	cmd[10] = GET_CHALLENGE | agid;

	u8	data[16];

	if (!mDev->SendCommandIn(cmd, data, 16))	return false;

	// get KEY2 from challenge
	CSS_CryptKey(variant, CSS_KEY2, data + 4, key);

	// send KEY2 to drive
	// do a SEND KEY
	cmd = SCSICmd12(SEND_KEY, 12);

	cmd[10] = SEND_KEY2 | agid;

	data[0] = 0;
	data[1] = 10;
	data[2] = 0;
	data[3] = 0;
	data[4] = key[0];
	data[5] = key[1];
	data[6] = key[2];
	data[7] = key[3];
	data[8] = key[4];
	data[9] = 0;
	data[10] = 0;
	data[11] = 0;

	if (!mDev->SendCommandOut(cmd, data, 12))	return false;

	return true;
}

// GetDiskKey
//
// get the disk key from the disk - this completes CSS authentication

bool SCSI_CDVD::GetDiskKey(int agid, u8 bus_key[5])
{
	SCSICmd12	cmd(READ_DVD_STRUCTURE, 2052);

	cmd[7] = GET_DISK_KEY;
	cmd[10] = agid;

	u8	data[2052];

	if (!mDev->SendCommandIn(cmd, data, 2052))	return false;

	for (int ii = 0; ii < 2048; ii++)	data[ii + 4] ^= bus_key[ii % 5];

	return true;
}
