// ASPI.h
//
// ASPI driver

#include "wnaspi32.h"
#include "scsidefs.h"
#include "mtfuji.h"

typedef unsigned char	u8;
typedef unsigned short	u16;
typedef unsigned long	u32;

class SCSI_CDVD;

// SCSI_Cmd
//
// SCSI command

class SCSICmd
{
public:
	SCSICmd(u8 length)		{ mLength = length; memset(mData, 0, length); }

	u8& operator[](int n)				{ return mData[n]; }
	const u8& operator[](int n) const	{ return mData[n]; }

	u8		GetLength() const			{ return mLength; }

protected:
	u8	mLength;
	u8	mData[12];
};

// SCSICmd6
// SCSICmd10
// SCSICmd12
//
// different-sized SCSI_Cmds

class SCSICmd6 : public SCSICmd
{
public:
	SCSICmd6(u8 command, u16 buflen) : SCSICmd(6)
	{
		mData[0] = command;
		mData[3] = buflen >> 8;
		mData[4] = buflen & 0xFF;
	}
};

class SCSICmd10 : public SCSICmd
{
public:
	SCSICmd10(u8 command, u16 buflen) : SCSICmd(10)
	{
		mData[0] = command;
		mData[7] = buflen >> 8;
		mData[8] = buflen & 0xFF;
	}
};

class SCSICmd12 : public SCSICmd
{
public:
	SCSICmd12(u8 command, u16 buflen) : SCSICmd(12)
	{
		mData[0] = command;
		mData[8] = buflen >> 8;
		mData[9] = buflen & 0xFF;
	}
};

// SCSIDevice
//
// a SCSI device

class SCSIDevice
{
public:
	// InitSCSI
	//
	// intialize SCSI libraray
	static bool		InitSCSI();

	// TermSCSI
	//
	// terminate SCSI library
	static void		TermSCSI();

	// GetNextDevice
	//
	// starting from prev = 0, go through list of devices
	static SCSIDevice*	GetNextDevice(SCSIDevice* prev = 0)		{ return prev ? prev->mNext : sFirstDevice; }

	// SendCommandOut
	//
	// send a command to the SCSI device and receive data back
	bool	SendCommandOut(const SCSICmd& cmd, void* dptr, u32 dlen)		{ return SendCommand(cmd, dptr, dlen, SRB_DIR_OUT); }

	// SendCommandIn
	//
	// send a command and data to the SCSI device
	bool	SendCommandIn(const SCSICmd& cmd, void* buffer, u32 buflen)		{ return SendCommand(cmd, buffer, buflen, SRB_DIR_IN); }

	// SendCommandNoData
	//
	// send a command with no data transfer
	bool	SendCommandNoData(const SCSICmd& cmd)							{ return SendCommand(cmd, 0, 0, 0); }

	// GetVendorId
	//
	// get vendor ID string
	const char*	GetVendorID()	{ return mVendorID; }

	SCSI_CDVD*	GetCDVD()		{ return mCDVD; }

private:
	SCSIDevice(int host_adapter, int id, int lun, u8 type);

	bool	SendCommand(const SCSICmd& cmd, void* buffer, u32 buflen, u8 flags);

	static	SCSIDevice*		sFirstDevice;
	static	HINSTANCE		sDLL;
	static	HANDLE			sSyncEvent;
	static	u32				(*sGetASPI32SupportInfo)();
	static	u32				(*sSendASPI32Command)(void*);

	int			mHostAdapter;		// host adapter #
	int			mID;				// SCSI ID
	int			mLUN;				// SCSI LUN
	u8			mDeviceType;		// SCSI device type
	char		mVendorID[51];		// vendor ID, 0-terminated
	SCSI_CDVD*	mCDVD;				// CDVD device, else 0

	SCSIDevice*	mNext;			// next device
};
