// ASPI.cpp
//
// ASPI driver

#include <stdafx.h>

#include "aspi.h"
#include "cdvd.h"

SCSIDevice*		SCSIDevice::sFirstDevice;
HINSTANCE		SCSIDevice::sDLL;
HANDLE			SCSIDevice::sSyncEvent;
u32				(*SCSIDevice::sGetASPI32SupportInfo)();
u32				(*SCSIDevice::sSendASPI32Command)(void*);

// InitSCSI
//
// initialize the SCSI library

bool SCSIDevice::InitSCSI()
{
	// load ASPI32 DLL
	sDLL = LoadLibrary("wnaspi32.dll");
	
	if (!sDLL)	return false;

	// clear list and create event
	sFirstDevice = 0;

	sSyncEvent = CreateEvent(0, true, false, 0);

	// get DLL functions
	sGetASPI32SupportInfo = (u32(*)())GetProcAddress(sDLL, "GetASPI32SupportInfo");
	sSendASPI32Command = (u32(*)(void*))GetProcAddress(sDLL, "SendASPI32Command");

	// get device count
	u32	status = sGetASPI32SupportInfo();

	if (((status >> 8) & 0xFF) != SS_COMP)	return false;

	int	adaptors = status & 0xFF;

	if (!adaptors)	return true;		// no error, just no adaptors!

	// scan devices
	SRB_GDEVBlock	cmd;

	for (int adaptor = 0; adaptor < adaptors; adaptor++)
	{
		for (int id = 0; id < 8; id++)
		{
			for (int lun = 0; lun < 8; lun++)
			{
				cmd.SRB_Cmd = SC_GET_DEV_TYPE;
				cmd.SRB_HaId = adaptor;
				cmd.SRB_Flags = 0;
				cmd.SRB_Hdr_Rsvd = 0;
				cmd.SRB_Target = id;
				cmd.SRB_Lun = lun;

				u32	result = sSendASPI32Command(&cmd);

				if (result == SS_COMP)
				{
					// found one!
					SCSIDevice*	dev = new SCSIDevice(adaptor, id, lun, cmd.SRB_DeviceType);

					dev->mNext = sFirstDevice;
					sFirstDevice = dev;
				}
			}
		}
	}

	return true;
}

// TermSCSI
//
// close the SCSI library

void SCSIDevice::TermSCSI()
{
	while (sFirstDevice)
	{
		SCSIDevice*	next = sFirstDevice->mNext;
		delete sFirstDevice;
		sFirstDevice = next;
	}

	CloseHandle(sSyncEvent);
	FreeLibrary(sDLL);
}

// new SCSIDevice
//
// initialize a new device object:
//   get its vendor ID
//   attach a SCSI_CDVD if appropriate

SCSIDevice::SCSIDevice(int host_adapter, int id, int lun, u8 type)
{
	mHostAdapter = host_adapter;
	mID = id;
	mLUN = lun;
	mDeviceType = type;
	mNext = 0;

	// perform Inquiry on self
	SCSICmd6	cmd(SCSI_INQUIRY, 58);
	u8			info[58];

	if (SendCommandIn(cmd, info, 58))
	{
		// get vendor ID
		memcpy(mVendorID, info + 8, 50);
		mVendorID[50] = 0;
	}

	// check CDVD stuff
	if (type == DTYPE_CROM)		mCDVD = new SCSI_CDVD(this);
	else						mCDVD = 0;
}

// SendCommand
//
// send a command to a SCSI device

bool SCSIDevice::SendCommand(const SCSICmd& cmd, void* buffer, u32 buflen, u8 flags)
{
	SRB_ExecSCSICmd srb;

	u8	cmdlen = cmd.GetLength();

	srb.SRB_Cmd = SC_EXEC_SCSI_CMD;
	srb.SRB_HaId = mHostAdapter;
	srb.SRB_Flags = SRB_EVENT_NOTIFY | flags;
	srb.SRB_Hdr_Rsvd = 0;
	srb.SRB_Target = mID;
	srb.SRB_Lun = mLUN;
	srb.SRB_BufLen = buflen;
	srb.SRB_BufPointer = (u8*)buffer;
	srb.SRB_SenseLen = SENSE_LEN;
	srb.SRB_CDBLen = cmdlen;
	srb.SRB_PostProc = (void(*)())sSyncEvent;

	for (int ii = 0; ii < cmdlen; ii++)
	{
		srb.CDBByte[ii] = cmd[ii];
	}

	ResetEvent(sSyncEvent);

	u32 result = sSendASPI32Command(&srb);

	if (result == SS_PENDING)	WaitForSingleObject(sSyncEvent, INFINITE);

	if (srb.SRB_Status != SS_COMP)	return false;

	return true;
}
